// -----------------------------------------------------------------------------
//    Automaton.mq4 Simple grid system.
//    Copyright (C) 2023 KAGA Hiroaki
//
//    This program is free software: you can redistribute it and/or modify
//    it under the terms of the GNU General Public License as published by
//    the Free Software Foundation, either version 3 of the License, or
//    (at your option) any later version.
//
//    This program is distributed in the hope that it will be useful,
//    but WITHOUT ANY WARRANTY; without even the implied warranty of
//    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//    GNU General Public License for more details.
//
//    You should have received a copy of the GNU General Public License
//    along with this program.  If not, see <https://www.gnu.org/licenses/>.
// -----------------------------------------------------------------------------

#property copyright "Copyright 2023, KAGA Hiroaki"
#property link      "https://59log.com" 
#property version   "1.06" 
#property strict

input int TradeType = 0; //TradeType 0:Long&Short 1:Long 2:Short
input bool Auto = false;

input double Lots = 0.01;
input double EntryStep = 100;
input double Ratio = 2.0;
input bool Martingale = true;

input double TakeProfit = 0;
input double StopLoss = 0;
input int Slippage = 10;
input double Spreads = 30;

input int BuyDepth = 2;
input int SellDepth = 2;
int BuyCount = 0;
int SellCount = 0;

input bool ServerTime = true;
input int StartHour = 0;
input int StartMinute = 0;
input int EndHour = 23;
input int EndMinute = 59;

input int Interval = 120;

bool StopBuy = false;
bool StopSell = false;

input int CutTime = 30;
input double Profit = 0;
bool BuyCutFlag = false;
bool SellCutFlag = false;

input double HighEquity = 0;
input double LowEquity = 0;
bool CloseAllFlag = false;
bool StopAll = false;

input bool Trend = false;
input bool TrendFollow = false;

// Bollinger Bands parameter
input ENUM_TIMEFRAMES BandsTimeFrame = PERIOD_CURRENT; // Bollinger Bands Time Frames
input int BandsPeriod = 20; // Bands Period
input double BandsDeviations = 2.0; // Bands Deviations
int BandsShift = 0;   // Bands Shift
ENUM_APPLIED_PRICE BandsAppliedPrice = PRICE_CLOSE;   // Bands Applied Price 0:close 1:open 2:high 3:low

input bool RiskOn = false;

input bool BarEntry = true;
input ENUM_TIMEFRAMES BarsTimeFrame = PERIOD_CURRENT;
int EntryBars = 0;
int CutBars = 0;

input bool Compound = false;
input double Leverage = 1;
input double MaxLot = 50;
input double LotRatio = 1000; // LotRatio Forex:1 Gold:1000 Bitcoin:100000

input bool DisplayButton = false;
string SellButtonID = "SellButton";
string BuyButtonID = "BuyButton"; 
string CloseButtonID = "CloseButton";

input int MagicNumber = 311;
input string Comments = "Automaton";

//+------------------------------------------------------------------+ 
//| Expert initialization function                                   | 
//+------------------------------------------------------------------+ 
int OnInit() 
   {
    if (StartHour < 0 || StartHour > 23) return INIT_FAILED;
    if (StartMinute < 0 || StartMinute > 59) return INIT_FAILED;
    if (EndHour < 0 || EndHour > 23) return INIT_FAILED;
    if (EndMinute < 0 || EndMinute > 59) return INIT_FAILED;

//--- Create button 
    if (DisplayButton)
       {
        ObjectCreate(0,SellButtonID,OBJ_BUTTON,0,100,100); 
        ObjectSetInteger(0,SellButtonID,OBJPROP_COLOR,clrWhite); 
        ObjectSetInteger(0,SellButtonID,OBJPROP_BGCOLOR,clrRed); 
        ObjectSetInteger(0,SellButtonID,OBJPROP_XDISTANCE,5); 
        ObjectSetInteger(0,SellButtonID,OBJPROP_YDISTANCE,15); 
        ObjectSetInteger(0,SellButtonID,OBJPROP_XSIZE,50); 
        ObjectSetInteger(0,SellButtonID,OBJPROP_YSIZE,30); 
        ObjectSetString(0,SellButtonID,OBJPROP_FONT,"Arial"); 
        ObjectSetString(0,SellButtonID,OBJPROP_TEXT,"Sell"); 
        ObjectSetInteger(0,SellButtonID,OBJPROP_FONTSIZE,10); 
        ObjectSetInteger(0,SellButtonID,OBJPROP_SELECTABLE,0); 

        ObjectCreate(0,BuyButtonID,OBJ_BUTTON,0,100,100); 
        ObjectSetInteger(0,BuyButtonID,OBJPROP_COLOR,clrWhite); 
        ObjectSetInteger(0,BuyButtonID,OBJPROP_BGCOLOR,clrBlue); 
        ObjectSetInteger(0,BuyButtonID,OBJPROP_XDISTANCE,65); 
        ObjectSetInteger(0,BuyButtonID,OBJPROP_YDISTANCE,15); 
        ObjectSetInteger(0,BuyButtonID,OBJPROP_XSIZE,50); 
        ObjectSetInteger(0,BuyButtonID,OBJPROP_YSIZE,30); 
        ObjectSetString(0,BuyButtonID,OBJPROP_FONT,"Arial"); 
        ObjectSetString(0,BuyButtonID,OBJPROP_TEXT,"Buy"); 
        ObjectSetInteger(0,BuyButtonID,OBJPROP_FONTSIZE,10); 
        ObjectSetInteger(0,BuyButtonID,OBJPROP_SELECTABLE,0); 

        ObjectCreate(0,CloseButtonID,OBJ_BUTTON,0,100,100); 
        ObjectSetInteger(0,CloseButtonID,OBJPROP_COLOR,clrBlack); 
        ObjectSetInteger(0,CloseButtonID,OBJPROP_BGCOLOR,clrGold); 
        ObjectSetInteger(0,CloseButtonID,OBJPROP_XDISTANCE,125); 
        ObjectSetInteger(0,CloseButtonID,OBJPROP_YDISTANCE,15); 
        ObjectSetInteger(0,CloseButtonID,OBJPROP_XSIZE,70); 
        ObjectSetInteger(0,CloseButtonID,OBJPROP_YSIZE,30); 
        ObjectSetString(0,CloseButtonID,OBJPROP_FONT,"Arial"); 
        ObjectSetString(0,CloseButtonID,OBJPROP_TEXT,"Close All"); 
        ObjectSetInteger(0,CloseButtonID,OBJPROP_FONTSIZE,10); 
        ObjectSetInteger(0,CloseButtonID,OBJPROP_SELECTABLE,0); 
       }

//--- 
    BuyCount = 0;
    SellCount = 0;
    StopBuy = false;
    StopSell = false;
    BuyCutFlag = false;
    SellCutFlag = false;
    CloseAllFlag = false;
    StopAll = false;
 
    return INIT_SUCCEEDED; 
   } 
  
//+------------------------------------------------------------------+ 
//| Expert deinitialization function                                 | 
//+------------------------------------------------------------------+ 
void OnDeinit(const int reason) 
   { 
//--- 
    ObjectDelete(0,SellButtonID); 
    ObjectDelete(0,BuyButtonID); 
    ObjectDelete(0,CloseButtonID);
   } 

//+------------------------------------------------------------------+ 
//| Expert tick function                                             | 
//+------------------------------------------------------------------+ 
void OnTick() 
   { 
//--- 
    // Position check
    int prevCount = BuyCount + SellCount;
    count_position();
    if (prevCount > (BuyCount + SellCount))
        close_check(prevCount - (BuyCount + SellCount));

    if (BuyCount == 0)
       {
        BuyCutFlag = false;       
        StopBuy = false;
       }
    else
       {
        if (BuyCutFlag)
           {
            close_buy();
            return;       
           }
       }
    if (SellCount == 0)
       {
        SellCutFlag = false;       
        StopSell = false;
       }
    else
       {
        if (SellCutFlag)
           {
            close_sell();
            return;       
           }
       }

    if (BuyCount == 0 && SellCount == 0)
       {
        CloseAllFlag = false;       
       }
    else
       {
        if (CloseAllFlag)
           {
            close_all_position();
            return;       
           }
       }

    MqlTick tick;
    SymbolInfoTick(_Symbol, tick);

    if (BuyCount > 0 || SellCount > 0)
       {
        cut(tick.time);
        equity();
       }

    int ret = order(tick.bid, tick.ask, tick.time);
   } 

//+------------------------------------------------------------------+
//| Position check function                                          |
//+------------------------------------------------------------------+
void close_check(int diff)
   {
    int count = 0;
    int total = OrdersHistoryTotal();
    for (int pos = total - 1; pos >= 0; pos--)
       {
        if (OrderSelect(pos, SELECT_BY_POS, MODE_HISTORY) == false) continue;
        if (OrderMagicNumber() == MagicNumber && _Symbol == OrderSymbol())
           {
            datetime closeTime = OrderCloseTime();
            if (closeTime > 0)
               {
                int order_type = OrderType();
                double profit = OrderProfit();
                if (profit < 0)
                   {
                    if (order_type == OP_BUY)
                        StopBuy = true;
                    if (order_type == OP_SELL)
                        StopSell = true;
                   }

                CutBars = Bars(NULL, PERIOD_CURRENT);

                count++; 
                if (count >= diff) return;
               }
           }
       }
   }

// Count position
void count_position()
   {
    BuyCount = 0;
    SellCount = 0;
    int total = OrdersTotal(); 
    for (int pos = 0; pos < total; pos++)
       {
        if (OrderSelect(pos, SELECT_BY_POS) == false) continue;
        if (OrderMagicNumber() == MagicNumber && _Symbol == OrderSymbol())
           {
            int order_type = OrderType();
            if (order_type == OP_BUY)
                BuyCount++;
            if (order_type == OP_SELL)
                SellCount++;
           }
       }
   }

//+------------------------------------------------------------------+ 
void OnChartEvent(const int id, 
                  const long &lparam, 
                  const double &dparam, 
                  const string &sparam) 
  { 
//--- Check the event by pressing a mouse button 
   if (id == CHARTEVENT_OBJECT_CLICK) 
      { 
       string clickedChartObject = sparam; 
       if (clickedChartObject == BuyButtonID) 
          { 
           // Buy Button pressed 
           ObjectSetInteger(0, BuyButtonID, OBJPROP_STATE, false); 
           buy(Bid, Ask);
          } 
       if (clickedChartObject == SellButtonID) 
          { 
           // Sell Button pressed 
           ObjectSetInteger(0, SellButtonID, OBJPROP_STATE, false); 
           sell(Bid, Ask);
          } 
       if (clickedChartObject == CloseButtonID) 
          { 
           // Close All Button pressed
           ObjectSetInteger(0, CloseButtonID, OBJPROP_STATE, false); 
           close_buy();
           close_sell();
           BuyCutFlag = true;
           SellCutFlag = true;
          }
      } 
  } 

//+------------------------------------------------------------------+
//| Order function                                                   |
//+------------------------------------------------------------------+
int order(double tickBid, double tickAsk, datetime tickTime)
   {
    if (StopAll) return 0;
    if (tickAsk > (tickBid + (Spreads * Point))) return 0;

    int ret = 0;
    datetime lasttime;
    int timeOffset;
    double range = EntryStep;
    if (range == 0) return 0;

    int cur_bars = Bars(NULL, PERIOD_CURRENT);
    if (cur_bars == CutBars) return 0;

    datetime dt;
    MqlDateTime curTime;
    if (ServerTime) dt = TimeCurrent(curTime);
    else dt = TimeLocal(curTime);
    string str_date = TimeToString(dt, TIME_DATE);

    string str_start = StringFormat("%s %02d:%02d:%02d", str_date, StartHour, StartMinute, 0);
    string str_end = StringFormat("%s %02d:%02d:%02d", str_date, EndHour, EndMinute, 59);

    datetime dt_start = StringToTime(str_start);
    datetime dt_end = StringToTime(str_end);

    if (BuyCount == 0 && Auto)
       {
        if (dt >= dt_start && dt <= dt_end)
           {
            if (Trend)
               {
                int type = bands(0, tickBid, tickAsk);
                if (type == 1)
                   {
                    ret = buy(tickBid, tickAsk);
                   }
               }
            else
               {
                ret = buy(tickBid, tickAsk);
               }
           }
       }
    if (BuyCount > 0)
       {
        cur_bars = Bars(NULL, BarsTimeFrame);
        if (!BarEntry || EntryBars != cur_bars)
           {
            lasttime = last_open_time(OP_BUY);
            timeOffset = (int)(tickTime - lasttime);
            if (lasttime > (datetime)0 && timeOffset > (Interval * 60))
               {
                int type = 1; 
                if (!RiskOn)
                   {
                    type = bands(1, tickBid, tickAsk); 
                   }
                if (type == 1)
                   {
                    range *= BuyCount;
                    type = entry(1, tickBid, tickAsk, range);
                    if (type == 1)
                       {
                        ret = buy(tickBid, tickAsk);
                       } 
                   }
               }
           }
       }
    
    if (SellCount == 0 && Auto)
       {
        if (dt >= dt_start && dt <= dt_end)
           {
            if (Trend)
               {
                int type = bands(0, tickBid, tickAsk);
                if (type == 2)
                   {
                    ret = sell(tickBid, tickAsk);
                   }
               }
            else
               {
                ret = sell(tickBid, tickAsk);
               }
           }
       }
    if (SellCount > 0)
       {
        cur_bars = Bars(NULL, BarsTimeFrame);
        if (!BarEntry || EntryBars != cur_bars)
           {
            lasttime = last_open_time(OP_BUY);
            timeOffset = (int)(tickTime - lasttime);
            if (lasttime > (datetime)0 && timeOffset > (Interval * 60))
               {
                int type = 2; 
                if (!RiskOn)
                   {
                    type = bands(2, tickBid, tickAsk); 
                   }
                if (type == 2)
                   {
                    range *= SellCount;
                    type = entry(2, tickBid, tickAsk, range);
                    if (type == 2)
                       {
                        ret = sell(tickBid, tickAsk);
                       } 
                   }
               }
           }
       } 

    return ret;
   }

// Buy order
int buy(double tickBid, double tickAsk)
   {
    if (TradeType != 0 && TradeType != 1) return 0;

    int ticket = 0;
    double sl = 0;
    double tp = 0;

    if (BuyCount < BuyDepth)
       {
        double order_price = tickAsk;
        int order_type = OP_BUY;

        if (TakeProfit > 0) tp = order_price + (TakeProfit * Point);
        if (StopLoss > 0) sl = order_price - (StopLoss * Point);

        double lot = Lots;
        if (Compound) lot = calc_lots();

        if (BuyCount > 0)
           {
            double ratio = 1;
            if (Ratio >= 1.0 && Ratio <= 3.0)
               {
                ratio = Ratio; 
               }
            if (Martingale)
               {
                lot *= MathPow(ratio, BuyCount);
               }
            else
               {
                lot *= (BuyCount + 1); 
               }
           }
        lot = adjust(lot);

        ticket = OrderSend(Symbol(), order_type, lot, order_price, Slippage, sl, tp,
                           Comments, MagicNumber, 0, Blue);
        if (ticket <= 0) Print("Error OrderSend function. Error code is ", GetLastError());
       }

    if (ticket > 0)
       {
        EntryBars = Bars(NULL, BarsTimeFrame);
        return ticket;
       }

    return 0;
   }

// Sell order
int sell(double tickBid, double tickAsk)
   {
    if (TradeType != 0 && TradeType != 2) return 0;
    if (StopSell) return 0;

    int ticket = 0;
    double sl = 0;
    double tp = 0;

    if (SellCount < SellDepth) 
       {
        double order_price = tickBid;
        int order_type = OP_SELL;

        if (TakeProfit > 0) tp = order_price - (TakeProfit * Point);
        if (StopLoss > 0) sl = order_price + (StopLoss * Point);   

        double lot = Lots;
        if (Compound) lot = calc_lots();

        if (SellCount > 0)
           {
            double ratio = 1;
            if (Ratio >= 1.0 && Ratio <= 3.0)
               {
                ratio = Ratio; 
               }
            if (Martingale)
               {
                lot *= MathPow(ratio, SellCount);
               }
            else
               {
                lot *= (SellCount + 1); 
               }
           }
        lot = adjust(lot);

        ticket = OrderSend(Symbol(), order_type, lot, order_price, Slippage, sl, tp,
                           Comments, MagicNumber, 0, Red);
        if (ticket <= 0) Print("Error OrderSend function. Error code is ", GetLastError());
       }

    if (ticket > 0)
       {
        EntryBars = Bars(NULL, BarsTimeFrame);
        return ticket;
       }
 
    return 0;
   }

double adjust(double val)
   {
    val *= 100;
    val = MathFloor(val);
    val /= 100;
    return val;
   }

// Calcurate lot
double calc_lots()
   {
    double lot_size = MarketInfo(Symbol(), MODE_LOTSIZE) * LotRatio;
    double lot_step = MarketInfo(Symbol(), MODE_LOTSTEP);
    double min_lot = MarketInfo(Symbol(), MODE_MINLOT);
    double max_lot = MarketInfo(Symbol(), MODE_MAXLOT);

    double n = (1 / lot_step);
    double lot = AccountEquity() * Leverage / lot_size;
    if (AccountCurrency() == "JPY")
       {
        lot /= 100;
       }
    lot *= n;
    lot = MathFloor(lot);
    lot /= n;
    if (lot < min_lot) lot = min_lot;
    if (lot > max_lot) lot = max_lot;
    if (lot > MaxLot) lot = MaxLot;
 
    return lot;
   }

// Cut position
void cut(datetime tickTime)
   {
    if (CutTime <= 0) return;

    double buy_profit = 0;
    double sell_profit = 0;
    int order_type = 0;
    datetime buy_time = (datetime)0;
    datetime sell_time = (datetime)0;
    int total = OrdersTotal(); 
    for (int pos = 0; pos < total; pos++)
       {
        if (OrderSelect(pos, SELECT_BY_POS) == false) continue;
        if (OrderMagicNumber() == MagicNumber && _Symbol == OrderSymbol())
           {
            order_type = OrderType();
            if (order_type != OP_BUY && order_type != OP_SELL) return; 

            if (order_type == OP_BUY)
               {
                buy_profit += OrderProfit();
                buy_profit += OrderSwap();
                buy_profit += OrderCommission();

                if (buy_time == (datetime)0)
                   {
                    buy_time = OrderOpenTime();
                   }
                else
                   {
                    if (buy_time > OrderOpenTime())
                       buy_time = OrderOpenTime();
                   }
               }

            if (order_type == OP_SELL)
               {
                sell_profit += OrderProfit();
                sell_profit += OrderSwap();
                sell_profit += OrderCommission();

                if (sell_time == (datetime)0)
                   {
                    sell_time = OrderOpenTime();
                   }
                else
                   {
                    if (sell_time > OrderOpenTime())
                       sell_time = OrderOpenTime();
                   }
               }
           }
       }

    if (BuyCount > 0)
       {
        if (buy_profit > Profit)
           {
            int timeOffset = (int)(tickTime - buy_time);
            if (CutTime > 0 && timeOffset > (CutTime * 60))
               {
                close_buy();
                BuyCutFlag = true;
               }
           }
       }

    if (SellCount > 0)
       {
        if (sell_profit > Profit)
           {
            int timeOffset = (int)(tickTime - sell_time);
            if (CutTime > 0 && timeOffset > (CutTime * 60))
               {
                close_sell();
                SellCutFlag = true;
               }
           }
       }
   }

// Close buy
void close_buy()
   {
    int total = OrdersTotal(); 
    for (int pos = total - 1; pos >= 0; pos--)
       {
        if (OrderSelect(pos, SELECT_BY_POS) == false) continue;
        if (OrderMagicNumber() == MagicNumber && _Symbol == OrderSymbol())
           {
            int order_type = OrderType();
            if (order_type == OP_BUY || order_type == OP_BUYSTOP || order_type == OP_BUYLIMIT)
                bool ret = OrderClose(OrderTicket(), OrderLots(), Bid, Slippage, Red);
           }
       }
   }

// Close sell
void close_sell()
   {
    int total = OrdersTotal(); 
    for (int pos = total - 1; pos >= 0; pos--)
       {
        if (OrderSelect(pos, SELECT_BY_POS) == false) continue;
        if (OrderMagicNumber() == MagicNumber && _Symbol == OrderSymbol())
           {
            int order_type = OrderType();
            if (order_type == OP_SELL || order_type == OP_SELLSTOP || order_type == OP_SELLLIMIT)
                bool ret = OrderClose(OrderTicket(), OrderLots(), Ask, Slippage, Blue);
           }
       }
   }

// Last open order time
datetime last_open_time(int type)
   {
    datetime last_time = (datetime)0;
    int total = OrdersTotal(); 
    for (int pos = 0; pos < total; pos++)
       {
        if (OrderSelect(pos, SELECT_BY_POS) == false) continue;
        if (OrderMagicNumber() == MagicNumber && _Symbol == OrderSymbol())
           {
            int order_type = OrderType();
            datetime open_time = OrderOpenTime();
            if (order_type == type)
               {
                if (last_time < open_time) last_time = open_time;
               }
           }
       }
    return last_time;
   }

// Account equity check
void equity()
   {
    double ae = AccountEquity();
    if ((HighEquity > 0 && ae >= HighEquity) || (LowEquity > 0 && ae <= LowEquity))
       {
        CloseAllFlag = true;
        StopAll = true;
        close_all_position();
       }
   }

// Close all position
void close_all_position()
   {
    bool ret = false;

    int total = OrdersTotal(); 
    for (int pos = total - 1; pos >= 0; pos--)
       {
        if (OrderSelect(pos, SELECT_BY_POS) == false) continue;
        int order_type = OrderType();
        if (order_type == OP_BUY)
            ret = OrderClose(OrderTicket(), OrderLots(), Bid, Slippage, Red);
        if (order_type == OP_SELL)
            ret = OrderClose(OrderTicket(), OrderLots(), Ask, Slippage, Blue);
       }
   }
   
//+------------------------------------------------------------------+
//| Order algorithm function                                      |
//+------------------------------------------------------------------+
// Entry 
int entry(int type, double tickBid, double tickAsk, double range)
   {
    double lastPrice = 0;
    double entryPrice = 0;

    // Buy
    if (type == 1)
       {
        lastPrice = buy_price();
        if (lastPrice == 0) return 0;
        entryPrice = lastPrice - range*Point;
        if (tickAsk < entryPrice) return 1;
       }

    // Sell
    if (type == 2)
       {
        lastPrice = sell_price();
        if (lastPrice == 0) return 0;
        entryPrice = lastPrice + range*Point;
        if (tickBid > entryPrice) return 2;
       }

    return 0;
   }

// Get buy position price.
double buy_price()
   {
    double lastPrice = 0;

    int total = OrdersTotal(); 
    for (int pos = 0; pos < total; pos++)
       {
        if (OrderSelect(pos, SELECT_BY_POS) == false) continue;
        if (OrderMagicNumber() == MagicNumber && _Symbol == OrderSymbol())
           {
            double openPrice = OrderOpenPrice();
            int order_type = OrderType();
            if (order_type == OP_BUY || order_type == OP_BUYSTOP || order_type == OP_BUYLIMIT)
               {
                if (lastPrice == 0 || lastPrice > openPrice)
                    lastPrice = openPrice;
               }
           }
       }

    return lastPrice;
   }

// Get sell position price.
double sell_price()
   {
    double lastPrice = 0;

    int total = OrdersTotal(); 
    for (int pos = 0; pos < total; pos++)
       {
        if (OrderSelect(pos, SELECT_BY_POS) == false) continue;
        if (OrderMagicNumber() == MagicNumber && _Symbol == OrderSymbol())
           {
            double openPrice = OrderOpenPrice();
            int order_type = OrderType();
            if (order_type == OP_SELL || order_type == OP_SELLSTOP || order_type == OP_SELLLIMIT)
               {
                if (lastPrice == 0 || lastPrice < openPrice)
                    lastPrice = openPrice;
               }
           }
       }

    return lastPrice;
   }

// Bollinger Bands
int bands(int type, double tickBid, double tickAsk)
   {
    double upper = iBands(NULL, BandsTimeFrame, BandsPeriod, BandsDeviations, BandsShift, BandsAppliedPrice, MODE_UPPER, 1);
    double lower = iBands(NULL, BandsTimeFrame, BandsPeriod, BandsDeviations, BandsShift, BandsAppliedPrice, MODE_LOWER, 1);

    double tickRate = (tickBid + tickAsk) / 2;

    if (type == 1)
       {
        if (tickRate > lower) return 1;  // over lower
       }
    else if (type == 2)
       {
        if (tickRate < upper) return 2;  // under upper
       }
    else
       {
        if (TrendFollow)
           {
            if (tickRate > upper) return 1;  // above buy
            else if (tickRate < lower) return 2;  // below sell
           }
        else
           {
            if (tickRate > upper) return 2;  // above sell
            else if (tickRate < lower) return 1;  // below buy
           }
       }

    return 0;
   }

//+------------------------------------------------------------------+
